import db from "../db.js";
import bcrypt from "bcryptjs";
import path from "path";
import fs from "fs";

// Add or update schedule
export const addUpdateSchedule = async (req, res) => {
  try {
    const doctorId = req.user.user_id;
    const { monday, tuesday, wednesday, thursday, friday, saturday, sunday } = req.body;

    // Check if schedule exists
    db.query("SELECT * FROM schedule WHERE doctor_id = ?", [doctorId], (err, result) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      const scheduleData = {
        monday: monday ? JSON.stringify(monday) : null,
        tuesday: tuesday ? JSON.stringify(tuesday) : null,
        wednesday: wednesday ? JSON.stringify(wednesday) : null,
        thursday: thursday ? JSON.stringify(thursday) : null,
        friday: friday ? JSON.stringify(friday) : null,
        saturday: saturday ? JSON.stringify(saturday) : null,
        sunday: sunday ? JSON.stringify(sunday) : null,
      };

      if (result.length > 0) {
        // Update existing schedule
        const sql = `
          UPDATE schedule 
          SET monday = ?, tuesday = ?, wednesday = ?, thursday = ?, friday = ?, saturday = ?, sunday = ?
          WHERE doctor_id = ?
        `;
        db.query(
          sql,
          [
            scheduleData.monday,
            scheduleData.tuesday,
            scheduleData.wednesday,
            scheduleData.thursday,
            scheduleData.friday,
            scheduleData.saturday,
            scheduleData.sunday,
            doctorId,
          ],
          (err, updateResult) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            res.status(200).json({
              message: "Schedule updated successfully",
              schedule_id: result[0].schedule_id,
            });
          }
        );
      } else {
        // Insert new schedule
        const sql = `
          INSERT INTO schedule (doctor_id, monday, tuesday, wednesday, thursday, friday, saturday, sunday)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        `;
        db.query(
          sql,
          [
            doctorId,
            scheduleData.monday,
            scheduleData.tuesday,
            scheduleData.wednesday,
            scheduleData.thursday,
            scheduleData.friday,
            scheduleData.saturday,
            scheduleData.sunday,
          ],
          (err, insertResult) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            res.status(201).json({
              message: "Schedule created successfully",
              schedule_id: insertResult.insertId,
            });
          }
        );
      }
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Get schedule
export const getSchedule = async (req, res) => {
  try {
    const doctorId = req.user.user_id;

    db.query("SELECT * FROM schedule WHERE doctor_id = ?", [doctorId], (err, result) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      if (result.length === 0) {
        return res.status(200).json({
          message: "No schedule found",
          schedule: null,
        });
      }

      const schedule = result[0];
      // Parse JSON fields
      const parsedSchedule = {
        ...schedule,
        monday: schedule.monday ? JSON.parse(schedule.monday) : null,
        tuesday: schedule.tuesday ? JSON.parse(schedule.tuesday) : null,
        wednesday: schedule.wednesday ? JSON.parse(schedule.wednesday) : null,
        thursday: schedule.thursday ? JSON.parse(schedule.thursday) : null,
        friday: schedule.friday ? JSON.parse(schedule.friday) : null,
        saturday: schedule.saturday ? JSON.parse(schedule.saturday) : null,
        sunday: schedule.sunday ? JSON.parse(schedule.sunday) : null,
      };

      res.status(200).json({
        message: "Schedule retrieved successfully",
        schedule: parsedSchedule,
      });
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Get all appointments for doctor
export const getMyAppointments = async (req, res) => {
  try {
    const doctorId = req.user.user_id;
    const { status } = req.query;

    let sql = `
      SELECT 
        a.appointment_id,
        a.date,
        a.time,
        a.status,
        a.notes,
        a.created_at,
        u.user_id as patient_id,
        u.first_name as patient_first_name,
        u.last_name as patient_last_name,
        u.phone as patient_phone,
        u.email as patient_email,
        u.profile as patient_profile
      FROM appointments a
      JOIN users u ON a.patient_id = u.user_id
      WHERE a.doctor_id = ?
    `;

    const params = [doctorId];

    if (status) {
      sql += " AND a.status = ?";
      params.push(status);
    }

    sql += " ORDER BY a.date DESC, a.time DESC";

    db.query(sql, params, (err, results) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      res.status(200).json({
        message: "Appointments retrieved successfully",
        appointments: results,
      });
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Accept pending appointment
export const acceptAppointment = async (req, res) => {
  try {
    const doctorId = req.user.user_id;
    const { appointmentId } = req.params;

    // Verify appointment belongs to doctor and is pending
    db.query(
      "SELECT * FROM appointments WHERE appointment_id = ? AND doctor_id = ? AND status = 'pending'",
      [appointmentId, doctorId],
      (err, result) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        if (result.length === 0) {
          return res.status(404).json({
            message: "Appointment not found, not yours, or already processed",
          });
        }

        // Update status to confirmed
        db.query(
          "UPDATE appointments SET status = 'confirmed' WHERE appointment_id = ?",
          [appointmentId],
          (err, updateResult) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            res.status(200).json({
              message: "Appointment accepted successfully",
            });
          }
        );
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Update appointment notes and status
export const updateAppointment = async (req, res) => {
  try {
    const doctorId = req.user.user_id;
    const { appointmentId } = req.params;
    const { notes, status } = req.body;

    if (!notes && !status) {
      return res.status(400).json({ message: "Either notes or status must be provided" });
    }

    // Validate status if provided
    if (status && !["pending", "confirmed", "completed", "cancelled"].includes(status)) {
      return res.status(400).json({ message: "Invalid status" });
    }

    // Verify appointment belongs to doctor
    db.query(
      "SELECT * FROM appointments WHERE appointment_id = ? AND doctor_id = ?",
      [appointmentId, doctorId],
      (err, result) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        if (result.length === 0) {
          return res.status(404).json({ message: "Appointment not found or not yours" });
        }

        // Build update query
        const updates = [];
        const values = [];

        if (notes !== undefined) {
          updates.push("notes = ?");
          values.push(notes);
        }
        if (status) {
          updates.push("status = ?");
          values.push(status);
        }

        values.push(appointmentId);

        const sql = `UPDATE appointments SET ${updates.join(", ")} WHERE appointment_id = ?`;

        db.query(sql, values, (err, updateResult) => {
          if (err) {
            return res.status(500).json({ message: "Database error", error: err });
          }

          res.status(200).json({
            message: "Appointment updated successfully",
          });
        });
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Update doctor profile
export const updateProfile = async (req, res) => {
  try {
    const doctorId = req.user.user_id;
    const {
      first_name,
      last_name,
      phone,
      address,
      education,
      speciality,
      bio,
      experience_years,
      fee,
      password,
    } = req.body;
    let profilePath = null;

    // If file uploaded, get the filename
    if (req.file) {
      profilePath = `users/${req.file.filename}`;
    }

    // Build update query dynamically
    const updates = [];
    const values = [];

    if (first_name) {
      updates.push("first_name = ?");
      values.push(first_name);
    }
    if (last_name) {
      updates.push("last_name = ?");
      values.push(last_name);
    }
    if (phone) {
      updates.push("phone = ?");
      values.push(phone);
    }
    if (address) {
      updates.push("address = ?");
      values.push(address);
    }
    if (education) {
      updates.push("education = ?");
      values.push(education);
    }
    if (speciality) {
      updates.push("speciality = ?");
      values.push(speciality);
    }
    if (bio) {
      updates.push("bio = ?");
      values.push(bio);
    }
    if (experience_years !== undefined) {
      updates.push("experience_years = ?");
      values.push(experience_years);
    }
    if (fee !== undefined) {
      updates.push("fee = ?");
      values.push(fee);
    }
    if (profilePath) {
      // Delete old profile if exists
      db.query("SELECT profile FROM users WHERE user_id = ?", [doctorId], (err, result) => {
        if (result && result[0] && result[0].profile) {
          const oldProfilePath = path.join(process.cwd(), "assets", result[0].profile);
          if (fs.existsSync(oldProfilePath)) {
            fs.unlinkSync(oldProfilePath);
          }
        }
      });

      updates.push("profile = ?");
      values.push(profilePath);
    }
    if (password) {
      const hashedPassword = await bcrypt.hash(password, 10);
      updates.push("password = ?");
      values.push(hashedPassword);
    }

    if (updates.length === 0) {
      return res.status(400).json({ message: "No fields to update" });
    }

    values.push(doctorId);

    const sql = `UPDATE users SET ${updates.join(", ")} WHERE user_id = ?`;

    db.query(sql, values, (err, result) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      // Get updated user
      db.query(
        "SELECT user_id, first_name, last_name, email, phone, address, education, speciality, bio, experience_years, fee, profile FROM users WHERE user_id = ?",
        [doctorId],
        (err, userResult) => {
          if (err) {
            return res.status(500).json({ message: "Database error", error: err });
          }

          res.status(200).json({
            message: "Profile updated successfully",
            user: userResult[0],
          });
        }
      );
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Get all reviews for doctor
export const getMyReviews = async (req, res) => {
  try {
    const doctorId = req.user.user_id;

    db.query(
      `SELECT 
        r.review_id,
        r.rating,
        r.comment,
        r.recommend,
        r.created_at,
        u.user_id as patient_id,
        u.first_name as patient_first_name,
        u.last_name as patient_last_name,
        u.profile as patient_profile,
        a.appointment_id,
        a.date as appointment_date
      FROM reviews r
      JOIN users u ON r.user_id = u.user_id
      JOIN appointments a ON r.appointment_id = a.appointment_id
      WHERE r.doctor_id = ?
      ORDER BY r.created_at DESC`,
      [doctorId],
      (err, results) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        // Get average rating
        db.query(
          `SELECT 
            AVG(rating) as average_rating,
            COUNT(*) as total_reviews
          FROM reviews
          WHERE doctor_id = ?`,
          [doctorId],
          (err, ratingResult) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            res.status(200).json({
              message: "Reviews retrieved successfully",
              average_rating: ratingResult[0].average_rating
                ? parseFloat(ratingResult[0].average_rating).toFixed(2)
                : "0.00",
              total_reviews: parseInt(ratingResult[0].total_reviews),
              reviews: results,
            });
          }
        );
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};


